#
# dbbackup.py  -  Methods for dumping and restoring database tables.
#
# Copyright (C) 2018 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
'''Required software:
- postgres-client (pg_dump, psql)
'''
import tempfile
import subprocess
import os
import argparse
from .generic import safe_syscall, load_module
from .dbconn import DBConn
from . import pfcodechg


DEFAULT_CONNSTR = 'dbname=perfactema user=postgres'


def connstr_to_dbargs(connstr):
    '''Convert connection string to dbargs dictionary.
    >>> (connstr_to_dbargs('dbname=perfactema user=zope') ==
    ...  {'dbname': 'perfactema', 'user': 'zope'})
    True
    '''
    return dict([pair.strip().split('=', 1) for pair in connstr.split(' ')])


def dbargs_to_connstr(dbargs):
    """Convert dbargs dict to connection string.
    >>> connstr_to_dbargs(
    ...     dbargs_to_connstr({'dbname': 'perfactema', 'user': 'zope'})
    ... ) == {'dbname': 'perfactema', 'user': 'zope'}
    True
    """
    return ' '.join([
        '{}={}'.format(key, value)
        for key, value in dbargs.items()
    ])


def dbargs_to_options(dbargs):
    '''Unfold dbargs into options for pg_dump.
    >>> dbargs_to_options({'dbname': 'perfactema', 'user': 'zope'})
    ['--dbname', 'perfactema', '--user', 'zope']
    '''
    return [i for k, v in sorted(dbargs.items()) for i in ['--'+k, v]]


def db_dump(tables=None, connstr=DEFAULT_CONNSTR):
    '''Dump data from indicated tables. Tables must be unique in the
    list of available schemata or given in schema.table syntax.

    The dump is produced and saved as a temporary file.  This method
    returns the path and file name for transmission to the client.

    '''
    if tables is None:
        tables = []
    for schema, table in tables:
        assert table.isalnum(), "Only alphanumeric table names allowed."

    dbargs = connstr_to_dbargs(connstr)

    tdir = tempfile.mkdtemp()
    fname = tdir + '/dump.sql'
    fh = open(fname, 'wb')
    fh.write('''-- PerFact DB dump
set session_replication_role to 'replica';
''')
    fh.close()
    for schema, table in tables:
        db_dump_table(fname, dbargs, schema, table, truncate=True)

    safe_syscall(['gzip', fname], raisemode=True)
    return tdir, 'dump.sql.gz'


def db_dump_table(fname, dbargs, schema=None, table=None, truncate=False,
                  append=True, inserts=True, database=None):
    '''
    Dump the table with an optional prepended truncate statement.
    Parameters:
      * fname: file name to be saved to
      * dbargs: a struct describing database connection arguments (see above)
      * schema: the name of the schema. If omitted, the table is looked for in
          the search path.
      * table: the name of the table to be dumped
      * truncate: if set, a statement to delete the table is included in front
          of the insert or copy statements.
      * append: if set (default), the statements are appended to the output
          file instead of overwriting it.
      * inserts: controls if insert statements (more explicit control naming
          all columns) or copy statements (more compact, ordered by id,
          diff-friendly) are generated.
      * database: If set, replaces the dbname argument in dbargs accordingly
    '''
    if schema is not None:
        table = schema+'.'+table

    fh = open(fname, 'ab' if append else 'wb')

    if truncate:
        fh.write('delete from {};\n'.format(table))
        fh.flush()
    if database is not None:
        # Create copy and update one key
        dbargs = dbargs.copy()
        dbargs['dbname'] = database
    opts = dbargs_to_options(dbargs)

    if inserts:
        cmd = ['pg_dump', '--column-inserts', '--data-only', '--table', table]
    else:
        fh.write((u"COPY %s FROM STDIN;\n" % table).encode('utf-8'))
        fh.flush()
        cmd = ['psql',
               '--command',
               "copy (select * from {table} order by {table}_id) to stdout"
               .format(table=table)
               ]
    cmd.extend(opts)
    c = subprocess.Popen(cmd, stdout=fh)
    c.wait()
    if not inserts:
        fh.write(u"\\.\n".encode('utf-8'))
    fh.close()


def db_dump_schema(fname, database, dbargs=None, opts=None):
    '''
    Dump the schema of a database to a file. Accepts either dbargs or directly
    opts.
    '''
    if opts is None:
        opts = dbargs_to_options(dbargs)
    subprocess.call(
        ['pg_dump',
         '--schema-only',
         '--file', fname,
         ] + opts
    )


def db_restore(infile, connstr=DEFAULT_CONNSTR):
    assert hasattr(infile, 'read'), "Needs a file-like object"
    origname = infile.filename.rsplit('/', 1)[-1]
    assert origname in ('dump.sql', 'dump.sql.gz'), "Invalid file name."

    tdir = tempfile.mkdtemp()
    fname = tdir + '/' + origname
    fh = open(fname, 'wb')
    while True:
        data = infile.read(65536)
        if not data:
            break
        fh.write(data)
    fh.close()

    if fname.endswith('.gz'):
        safe_syscall(['gunzip', fname], raisemode=True)
        fname = fname[:-3]

    dbargs = connstr_to_dbargs(connstr)
    opts = dbargs_to_options(dbargs)
    cmd = ['psql', '--single-transaction', '--file', fname]
    cmd = cmd + opts
    return safe_syscall(cmd, raisemode=True)


def git_snapshot(config, connstr=DEFAULT_CONNSTR):
    '''
    Create a git snapshot of schemata and tables configured in config.
    config should be a namespace containing the following names:
    * base_dir: the path to the git repo
    * subdir (default: __psql__): subdir used in base_dir. May be empty.
    * databases: a list of databases
    * db_tables: a dict mapping each database to a list of tables
    For each database, the schema as well as the listed tables are dumped.
    Afterwards, a git commit is performed.
    '''
    basedir = config.base_dir

    # if not set, subdir is __psql__, for backward compatibility (the same
    # config can be used as with zoperecord
    # Newer configs set subdir to ''
    subdir = getattr(config, 'subdir', '__psql__')
    if subdir:
        basedir = basedir + '/' + subdir

    try:
        os.stat(basedir)
    except OSError:
        print('Will create new directory %s' % basedir)
        os.makedirs(basedir)

    connstr = getattr(config, 'connstr', connstr)
    dbargs = connstr_to_dbargs(connstr)
    for database in config.databases:
        dbargs['dbname'] = database
        opts = dbargs_to_options(dbargs)
        connstr = dbargs_to_connstr(dbargs)

        print('Will dump database schema for %s' % database)
        db_dump_schema(
            fname='{basedir}/schema-{database}.sql'.format(
                basedir=basedir,
                database=database
            ),
            database=database,
            opts=opts
        )

        path = basedir+'/data-'+database
        try:
            os.stat(path)
        except OSError:
            print('Will create new directory %s' % path)
            os.makedirs(path)

        tables = config.db_tables.get(database, [])

        dbconn = DBConn(dbconn_string=connstr)
        dbconn.execute(
            "select tablename from pg_tables"
            " where tablename = any(%(tables)s)",
            tables=tables,
        )
        existing = {row[0] for row in dbconn.tuples()}
        dbconn.disconnect()

        for table in tables:
            if table not in existing:
                print("Skipping not existing table {table} in DB"
                      " {database}".format(table=table, database=database)
                      )
                continue
            print('Will dump database table %s / %s' % (database, table))
            db_dump_table(
                fname=path+'/'+table+'.sql',
                dbargs=dbargs,
                table=table,
                truncate=False,
                append=False,
                inserts=False,
            )
    pfcodechg.git_snapshot(
        directory=config.base_dir,
        commit_message=config.commit_message,
        subdir=subdir
    )


def binwrapper_dbrecord(default_config):
    """Wrapper for call from perfact-dbrecord or perfact-dbrecord3. Reads in
    config and custom config before calling git_snapshot."""
    parser = argparse.ArgumentParser(
        description="Record the PSQL database",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    parser.add_argument(
        '--config', '-c',
        type=str,
        default=default_config,
        help='Path to config'
    )
    parser.add_argument(
        '--custom-config',
        type=str,
        help='Path to custom config'
    )
    args = parser.parse_args()
    custom_config_fname = args.custom_config or (
        os.path.dirname(args.config) + '/dbrecord-custom.py'
    )
    config = load_module(args.config)
    if os.path.isfile(custom_config_fname):
        custom_config = load_module(custom_config_fname)
        if hasattr(custom_config, 'connstr'):
            config.connstr = custom_config.connstr
        if hasattr(custom_config, 'add_tables'):
            for dbname, tables in custom_config.add_tables.items():
                if dbname not in config.db_tables:
                    config.db_tables[dbname] = []
                config.db_tables[dbname].extend(tables)

    git_snapshot(config)


# Testing
if __name__ == '__main__':
    import doctest
    doctest.testmod()
